import { useState, useCallback } from 'react';

export function useHistory(initialState) {
    const [state, setState] = useState(initialState);
    const [past, setPast] = useState([]);
    const [future, setFuture] = useState([]);

    const set = useCallback((newState) => {
        setPast((prev) => {
            const newPast = [...prev, state];
            if (newPast.length > 50) newPast.shift(); // Limit history to 50
            return newPast;
        });
        setState(newState);
        setFuture([]);
    }, [state]);

    const undo = useCallback(() => {
        if (past.length === 0) return;
        const previous = past[past.length - 1];
        const newPast = past.slice(0, past.length - 1);

        setFuture((prev) => [state, ...prev]);
        setState(previous);
        setPast(newPast);
    }, [past, state]);

    const redo = useCallback(() => {
        if (future.length === 0) return;
        const next = future[0];
        const newFuture = future.slice(1);

        setPast((prev) => [...prev, state]);
        setState(next);
        setFuture(newFuture);
    }, [future, state]);

    const reset = useCallback((newState) => {
        setState(newState);
        setPast([]);
        setFuture([]);
    }, []);

    const setSilent = useCallback((newState) => {
        setState(newState);
    }, []);

    return {
        state,
        set,
        undo,
        redo,
        canUndo: past.length > 0,
        canRedo: future.length > 0,
        reset,
        setSilent
    };
}
